<?php
/*
 * @package   bfNetwork
 * @copyright Copyright (C) 2011,2012,2013,2014,2015,2016,2017,2018,2019,2020,2021,2022 Blue Flame Digital Solutions Ltd. All rights reserved.
 * @license   GNU General Public License version 3 or later
 *
 * @see       https://mySites.guru/
 * @see       https://www.phil-taylor.com/
 *
 * @author    Phil Taylor / Blue Flame Digital Solutions Limited.
 *
 * bfNetwork is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * bfNetwork is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this package.  If not, see http://www.gnu.org/licenses/
 *
 * If you have any questions regarding this code, please contact phil@phil-taylor.com
 */

// Decrypt or die
require 'bfEncrypt.php';

/*
 * If we have got here then we have already passed through decrypting
 * the encrypted header and so we are sure we are now secure and no one
 * else cannot run the code below......
 */

class bfUpgrade
{
    /**
     * We pass the command to run as a simple integer in our encrypted request this is mainly to speed up the decryption
     * process, plus its a single digit(or 2) rather than a huge string to remember :-).
     */
    private $_methods = array(
        1 => 'getAllUpdates',
        2 => 'doUpgrade',
        3 => 'createRestorationFile',
        4 => 'extractUpdate',
        5 => 'finishup',
    );

    /**
     * I inject the request to the object.
     *
     * @param stdClass $dataObj
     */
    public function __construct($dataObj)
    {
        // init WordPress
        require 'bfInitWordpress.php';

        require_once ABSPATH . 'wp-admin/includes/file.php';
        require_once ABSPATH . 'wp-admin/includes/class-wp-upgrader.php';
        require_once ABSPATH . 'wp-admin/includes/update.php';
        require_once ABSPATH . 'wp-admin/includes/plugin.php';

        // Set the request vars
        $this->_dataObj = $dataObj;
    }

    /**
     * I'm the controller - I run methods based on the request integer.
     */
    public function run()
    {
        if (property_exists($this->_dataObj, 'c')) {
            $c = (int) $this->_dataObj->c;
            if (array_key_exists($c, $this->_methods)) {
                // call the right method
                $this->{$this->_methods[$c]} ();
            } else {
                // Die if an unknown function
                bfEncrypt::reply('error', 'No Such method #err1 - ' . $c);
            }
        } else {
            // Die if an unknown function
            bfEncrypt::reply('error', 'No Such method #err2');
        }
    }

    /**
     * Returns the (cached) list of updates for every section: installed version, current branch updates, sts/lts
     * updates, testing updates.
     *
     * @param bool $force Should I forcibly reload the update information, refreshing the cache?
     */
    public function getAllUpdates($force = true)
    {
        wp_version_check(array(), $force);

        $updateInfo = (array) get_core_updates();
        $data       = $updateInfo[0];

        $data->Currently_Installed_Version = get_bloginfo('version');
        $data->installed                   = get_bloginfo('version');
        $data->infourl                     = null; // @todo
        $data->downloadurl                 = $data->download;
        $data->PHP_VERSION                 = \PHP_VERSION;
        $data->hasAkeebaBackup             = is_plugin_active('akeebabackupwp/akeebabackupwp.php');
        $data->lastBackupDetails           = null;

        // Backward compatibility with workers' Job ... fml...
        $obj      = new \stdClass();
        $obj->{0} = $data;

        bfEncrypt::reply('success', array(
            'msg' => json_encode($obj),
        ));
    }

    /**
     * I'm the controller - I run methods based on the request integer.
     */
    public function doUpgrade()
    {
        delete_option('core_updater.lock');

        wp_version_check(array(), true);

        $easyUpdate = false;

        if (true === $easyUpdate) {
            $offer = find_core_auto_update();
        } else {
            $offer = get_core_updates()[0];
        }

        $skin      = new Automatic_Upgrader_Skin();
        $upgrader2 = new Core_Upgrader($skin);
        $res       = $upgrader2->upgrade($offer, array(
            'allow_relaxed_file_ownership' => true,
        ));

        if (! is_string($res) && 'WP_Error' == get_class($res)) {
            $ret             = array();
            $ret['status']   = 'error';
            $ret['messages'] = $res->get_error_messages();
            $res             = $ret;
        } else {
            $ret['status']  = 'success';
            $ret['version'] = $res;
            $res            = $ret;
        }

        bfEncrypt::reply($ret['status'], $res);
    }
}

// init this class
$upgradeController = new bfUpgrade($dataObj);

$upgradeController->run();
